/* --------------------------------------------------------------
  FilterBox.js 2019-06-07
  Gambio GmbH
  http://www.gambio.de
  Copyright (c) 2019 Gambio GmbH
  Released under the GNU General Public License (Version 2)
  [http://www.gnu.org/licenses/gpl-2.0.html]
  --------------------------------------------------------------*/


'use strict';

/**
 * Widget Constructor
 *
 * @param {jQuery} $target The selector of the element that will contain the widget.
 * @param {jQuery} $settingsList The <ul> element that contains the list of the settings.
 * @param {jQuery} $container The parent container element, where the PerfectScrollbar is bound.
 * @param {StyleConfiguration} styleConfiguration Contains the style configuration data.
 */
export default class FilterBox {
	/**
	 * Class Constructor
	 * 
	 * @param {jQuery} $target
	 * @param {jQuery} $settingsList
	 * @param {jQuery} $container
	 * @param {StyleConfiguration} styleConfiguration
	 */
	constructor($target, $settingsList, $container, styleConfiguration) {
		StyleEdit.Validator.isObject($target);
		StyleEdit.Validator.isObject($settingsList);
		StyleEdit.Validator.isObject($container);
		StyleEdit.Validator.isObject(styleConfiguration);
		
		/**
		 * Target Selector
		 *
		 * @type {jQuery}
		 */
		this.$target = $target;
		
		/**
		 * Setting List Selector
		 *
		 * @type {jQuery}
		 */
		this.$settingsList = $settingsList;
		
		/**
		 * Container Selector
		 *
		 * @type {jQuery}
		 */
		this.$settingsList = $container;
		
		/**
		 * Style Configuration Instance
		 *
		 * @type {StyleConfiguration}
		 */
		this.styleConfiguration = styleConfiguration;
		
		/**
		 * Contains the available filter tags.
		 *
		 * @type {String[]}
		 */
		this.availableTags = [];
	}
	
	/**
	 * Initialize Widget
	 */
	initialize() {
		// Parse the style configuration filter tags.
		this._parseStyleTags();
		
		// Load the widget template. 
		this._loadWidgetTemplate();
		
		// Bind the event handlers.
		this._bindEventHandlers();
	}
	
	/**
	 * Bind Event Handlers
	 *
	 * @private
	 */
	_bindEventHandlers() {
		this.$target.on('click', '.action', event => this._onFilterItemClick(event));
	}
	
	/**
	 * Load Widget Template
	 *
	 * @private
	 */
	_loadWidgetTemplate() {
		const template = $('#widget-filter-box-template').html();
		const data = {tags: []};
		
		data.tags.push({
			value: 'filter-display-all',
			text: StyleEdit.Language.translate('label_filter_display_all', 'style_edit')
		});
		
		$.each(this.availableTags, (index, tag) => {
			data.tags.push({
				value: tag,
				text: StyleEdit.Language.translate(tag, 'template')
			});
		});
		
		this.$target.html(Mustache.render(template, data));
		
		// Add material design effects to new objects. 
		$.material.init();
	}
	
	/**
	 * Event: On Filter Item Click
	 *
	 * @param event {jQuery.Event}
	 *
	 * @private
	 */
	_onFilterItemClick(event) {
		const $target = $(event.currentTarget); 
		const currentTag = $target.attr('data-tag');
		
		// Check for display all option.
		if (currentTag === 'filter-display-all') {
			this.$target.find('.filter')
				.removeClass('active')
				.find('.info')
				.text('Filter');
			
			this.$settingsList
				.find('.option .form-group')
				.show();
		} else {
			this.$target.find('.filter')
				.addClass('active')
				.find('.info')
				.text($target.text());
			
			this.$settingsList.find('.option .form-group').each((index, formGroup) => {
				if ($(formGroup).attr('data-tags') === undefined) { // The form-group has no tags so hide it.
					$(formGroup).hide();
					return true; // continue
				}
				
				const formGroupTags = $(formGroup).attr('data-tags').split('|');
				
				if (formGroupTags.indexOf(currentTag) > -1) {
					$(formGroup).show();
				} else {
					$(formGroup).hide();
				}
			});
		}
		
		// Hide the setting groups that do not have any element visible.
		this.$settingsList.find('.option').show();
		
		this.$settingsList.find('.option').each((index, option) => {
			let hideOptionRow = true;
			
			$(option).find('.dropdown-menu .form-group').each((index, formGroup) => {
				if ($(formGroup).css('display') !== 'none') {
					hideOptionRow = false;
					
					// Display the .title rows that reside bellow the displayed elements. This workaround will
					// ensure that he Title widget instances will be displayed correctly while filtering the settings. 
					$(formGroup).prevUntil('.option').find('.title').last().parents('.form-group').show();
					
					return false;
				}
			});
			
			if (hideOptionRow === true) {
				$(option).hide();
			}
		});
		
		// Trigger a change event on the .image-list select boxes in order to show/hide the image-related settings. 
		this.$settingsList.find('.image-list').trigger('image:toggle_dependencies');
		
		// Refresh container scrollbars. 
		this.$settingsList.perfectScrollbar('update');
	}
	
	/**
	 * Parse the style configuration tags.
	 *
	 * This method will walk over all the style settings and find the unique filtering
	 * tags that will be used for the setting filtering.
	 *
	 * @private
	 */
	_parseStyleTags() {
		$.each(this.styleConfiguration.getSettings(), (index, collection) => {
			$.each(collection.getEntries(), (index, setting) => {
				if (setting.getTags()) {
					$.each(setting.getTags(), (index, tag) => {
						if (this.availableTags.indexOf(tag) === -1) {
							this.availableTags.push(tag);
						}
					});
				}
			});
		});
	}
}
	